/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Framework Error Definitions
::   Copyright   :   (C)2003-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file FrameworkError.h
    The framework has numerous errors with context sensitive information. They can be treated as pass or fail
    using a simple signed check or the value can be examined against a specific result

    All errors are defined to fit into the signed integer format for the chosen platform. The error code
    is constructed with a number of fields. The size of the fields and their members is dependant upon
    what the platform's native integer size is and how the information is to be employed.

    The SUCCESS and FAILURE macros should always be used to test an error.
*/

#ifndef __FRAMEWORKERROR_H
#define __FRAMEWORKERROR_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <NativeError.h>
#include <BitFunctions.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/
#pragma pack(1)

/* Name: E_NativeFWError ClassID:ENUMDEC EnumDflt:"Illegal Error" */
typedef enum
{
    NATIVEFWERROR_NONE,              /* EnumTxt:"Normal Operation"*/
    BAD_RESOURCE_ID,                 /* EnumTxt:"Bad Resource ID"*/                    /* 0x01 */
    HARDWARE_NOT_SUPPORTED,          /* EnumTxt:"Hardware Not Supported"*/             /* 0x02 */
    BEHAVIOUR_NOT_SUPPORTED,         /* EnumTxt:"Behaviour Not Supported"*/            /* 0x03 */
    BAD_ATTRIBUTE_DATA,              /* EnumTxt:"Bad Attribute Data"*/                 /* 0x04 */
    FAIL_ON_CREATE,                  /* EnumTxt:"Fail On Create"*/                     /* 0x05 */
    FAIL_ON_SET_ATTRIBUTE,           /* EnumTxt:"Fail On Set Attributes"*/             /* 0x06 */
    RESOURCE_NOT_CREATED,            /* EnumTxt:"Resource Not Created"*/               /* 0x07 */
    MISSING_ATTRIBUTE,               /* EnumTxt:"Missing Attribute"*/                  /* 0x08 */
    BEHAVIOUR_METHOD_UNSUPPORTED,    /* EnumTxt:"Behaviour Method Not Supported"*/     /* 0x09 */
    FAIL_ON_OPERATION,               /* EnumTxt:"A Service Failed"*/                   /* 0x0A */
    FAIL_ON_DESTROY,                 /* EnumTxt:"Failed to destroy a resource"*/       /* 0x0B */
    CREATED_RESOURCE,                /* EnumTxt:"Resource Was Created"*/               /* 0x0C */
    RESOURCE_CONFLICT,               /* EnumTxt:"Resource Conflict"*/                  /* 0x0D */
    UNKNOWN_CRANK_ENCODER,           /* EnumTxt:"Invalid Encoder Type"*/               /* 0x0E */
    ENCODER_2STROKE_UNSUPPORTED,     /* EnumTxt:"2-Strike Not Supported"*/             /* 0x0F */
    ENCODER_720_UNSUPPORTED,         /* EnumTxt:"720degCA Pattern Not Supported"*/     /* 0x10 */
    UNSUPPORTED_CAM_ENCODER,         /* EnumTxt:"CAM Pattern Not Supported"*/          /* 0x11 */
    UNKNOWN_CAM_ENCODER,             /* EnumTxt:"Unknown CAM Pattern Supplied"*/       /* 0x12 */
    NOT_ZERO_SPEED,                  /* EnumTxt:"Engine Speed Present"*/               /* 0x13 */
    HW_INTERFACE_NOT_SUPPORT,        /* EnumTxt:"Hardware Interface Not Supported"*/   /* 0x14 */
    SERVICE_ALREADY_STARTED,         /* EnumTxt:"Service Already Started"*/            /* 0x15 */
    REGISTER_WRITE_VERIFY_FAILED,    /* EnumTxt:"Register Write Failed to verify" */   /* 0x16 */
    REGISTER_ACCESS_DENIED,          /* EnumTxt:"Register access was denied" */        /* 0x17 */
    OUT_OF_MEMORY,                   /* EnumTxt:"Out of free store memory" */          /* 0x18 */
    SW_Q_OVERRUN,                    /* EnumTxt:"Software Q Overrun"*/                 /* 0x19 */
    HW_Q_OVERRUN,                    /* EnumTxt:"Hardware Q Overrun"*/                 /* 0x1A */
    UNSUPPORTED_IN_ARCHIVE,          /* EnumTxt:"Feature unsupported in archive" */    /* 0x1B */
    OUT_OF_RESOURCE,                 /* EnumTxt:"Insufficient FW resource" */          /* 0x1C */
    OS_RESOURCE_CONFLICT,            /* EnumTxt:"OS Resource conflict" */              /* 0x1D */
    DRIVER_CONFLICT,                 /* EnumTxt:"Driver Mode conflict" */              /* 0x1E */
    RTC_TIME_ILLEGAL,                /* EnumTxt:"RTC time is illegal" */               /* 0x1F */
    FFS_UNMOUNTED,                   /* EnumTxt:"FFS unmounted" */                     /* 0x20 */
    ACCESS_FROM_ISR,                 /* EnumTxt:"Access from an ISR" */                /* 0x21 */
    FFS_INTERNAL,                    /* EnumTxt:"FFS internal" */                      /* 0x22 */
    REQUEST_TIMEOUT,                 /* EnumTxt:"Operation timed out" */               /* 0x23 */
    TASK_KERNEL_API,                 /* EnumTxt:"Task Kernel API Usage" */             /* 0x24 */
    REQUEST_ABORTED,                 /* EnumTxt:"Operation aborted" */                 /* 0x25 */
    FILE_NOT_FOUND,                  /* EnumTxt:"File not Found" */                    /* 0x26 */
    FILENAME_INVALID,                /* EnumTxt:"Invalid characters in filename" */    /* 0x27 */
    FILENAME_NOT_UNIQUE,             /* EnumTxt:"Duplicate filename" */                /* 0x28 */
    FILENAME_TOO_LONG,               /* EnumTxt:"Filename is too long" */              /* 0x29 */
    DIRECTORY_NOT_EMPTY,             /* EnumTxt:"Dir not empty on remove" */           /* 0x2A */
    DIRECTORY_NAME_NOT_FOUND,        /* EnumTxt:"Dir not found" */                     /* 0x2B */
    DIRECTORY_INVALID,               /* EnumTxt:"Dir name invalid" */                  /* 0x2C */
    DIRECTORY_EXIST,                 /* EnumTxt:"Dir exists" */                        /* 0x2D */
    DIRECTORY_TOO_DEEP,              /* EnumTxt:"Dir too deep" */                      /* 0x2E */
    BAD_FW_HANDLE,                   /* EnumTxt:"Illegal Handle" */                    /* 0x2F */
    FS_EOF,                          /* EnumTxt:"End of File" */                       /* 0x30 */
    FS_MEDIA_FULL,                   /* EnumTxt:"Media Full" */                        /* 0x31 */
    FS_MEDIA_WRITE,                  /* EnumTxt:"Media Write" */                       /* 0x32 */
    FS_MEDIA_READ,                   /* EnumTxt:"Media Read" */                        /* 0x33 */
    FS_MEDIA_WRITE_PROTECT,          /* EnumTxt:"Media Write Protected" */             /* 0x34 */
    FS_FILE_READ_ONLY,               /* EnumTxt:"File Read Only" */                    /* 0x35 */
    FS_MEDIA_CHANGED,                /* EnumTxt:"FS Media Changed" */                  /* 0x36 */
    USB_INTERNAL,                    /* EnumTxt:"Internal USB" */                      /* 0x37 */
    USB_HIGH_VOLTAGE,                /* EnumTxt:"Voltage too high for USB IC" */       /* 0x38 */
    ENCODER_NOT_CREATED,             /* EnumTxt:"Encoder not created" */               /* 0x39 */

/* 0x40 to 0x5F reserved for the TPU */
    TPU_FW_RESOURCE_CONFLICT = 0x40, /* EnumTxt:"TPU Chan Conflict" */                 /* 0x40 */
    TPU_FW_FUNC_NOT_MASK,            /* EnumTxt:"TPU Func Unavailable" */              /* 0x41 */
    TPU_FW_CONFIG_UNSUPPORTED,       /* EnumTxt:"TPU Config" */                        /* 0x42 */
    TPU_FW_USAGE,                    /* EnumTxt:"TPU Primative Usage" */               /* 0x43 */
    TPU_FW_PRIMITIVE_CNT_EXCEEDED,   /* EnumTxt:"TPU Primitive Count" */               /* 0x44 */
    TPU_FW_RESTRICT_PARAM,           /* EnumTxt:"TPU Restricted TPU channel" */        /* 0x45 */
    TPU_FW_NOT_CREATED,              /* EnumTxt:"TPU Not Created" */                   /* 0x46 */
    TPU_FW_SM_CONST_ERROR,           /* EnumTxt:"TPU Self Modify constant write err" *//* 0x47 */

/* Overwrite TPU_UNDEFINED_ERROR entries as Errors get defined in TPU\Generic\API\TPUuCodeError.h */
    TPU_FW_UNDEFINED_ERROR_0x48,     /* EnumTxt:"TPU Error 0x48" */                    /* 0x48 */
    TPU_FW_UNDEFINED_ERROR_0x49,     /* EnumTxt:"TPU Error 0x49" */                    /* 0x49 */
    TPU_FW_UNDEFINED_ERROR_0x4A,     /* EnumTxt:"TPU Error 0x4A" */                    /* 0x4A */
    TPU_FW_UNDEFINED_ERROR_0x4B,     /* EnumTxt:"TPU Error 0x4B" */                    /* 0x4B */
    TPU_FW_UNDEFINED_ERROR_0x4C,     /* EnumTxt:"TPU Error 0x4C" */                    /* 0x4C */
    TPU_FW_UNDEFINED_ERROR_0x4D,     /* EnumTxt:"TPU Error 0x4D" */                    /* 0x4D */
    TPU_FW_UNDEFINED_ERROR_0x4E,     /* EnumTxt:"TPU Error 0x4E" */                    /* 0x4E */
    TPU_FW_UNDEFINED_ERROR_0x4F,     /* EnumTxt:"TPU Error 0x4F" */                    /* 0x4F */
    TPU_FW_UNDEFINED_ERROR_0x50,     /* EnumTxt:"TPU Error 0x50" */                    /* 0x50 */
    TPU_FW_UNDEFINED_ERROR_0x51,     /* EnumTxt:"TPU Error 0x51" */                    /* 0x51 */
    TPU_FW_UNDEFINED_ERROR_0x52,     /* EnumTxt:"TPU Error 0x52" */                    /* 0x52 */
    TPU_FW_UNDEFINED_ERROR_0x53,     /* EnumTxt:"TPU Error 0x53" */                    /* 0x53 */
    TPU_FW_UNDEFINED_ERROR_0x54,     /* EnumTxt:"TPU Error 0x54" */                    /* 0x54 */
    TPU_FW_UNDEFINED_ERROR_0x55,     /* EnumTxt:"TPU Error 0x55" */                    /* 0x55 */
    TPU_FW_UNDEFINED_ERROR_0x56,     /* EnumTxt:"TPU Error 0x56" */                    /* 0x56 */
    TPU_FW_UNDEFINED_ERROR_0x57,     /* EnumTxt:"TPU Error 0x57" */                    /* 0x57 */
    TPU_FW_UNDEFINED_ERROR_0x58,     /* EnumTxt:"TPU Error 0x58" */                    /* 0x58 */
    TPU_FW_UNDEFINED_ERROR_0x59,     /* EnumTxt:"TPU Error 0x59" */                    /* 0x59 */
    TPU_FW_UNDEFINED_ERROR_0x5A,     /* EnumTxt:"TPU Error 0x5A" */                    /* 0x5A */
    TPU_FW_UNDEFINED_ERROR_0x5B,     /* EnumTxt:"TPU Error 0x5B" */                    /* 0x5B */
    TPU_FW_UNDEFINED_ERROR_0x5C,     /* EnumTxt:"TPU Error 0x5C" */                    /* 0x5C */
    TPU_FW_UNDEFINED_ERROR_0x5D,     /* EnumTxt:"TPU Error 0x5D" */                    /* 0x5D */
    TPU_FW_UNDEFINED_ERROR_0x5E,     /* EnumTxt:"TPU Error 0x5E" */                    /* 0x5E */
    TPU_FW_UNDEFINED_ERROR_0x5F,     /* EnumTxt:"TPU Error 0x5F" */                    /* 0x5F */
/* 0x5F End of TPU Reservation       */

    NUM_NATIVE_FW_ERRORS
} E_NativeFWError;

/* Name: E_NativeFWSuccess ClassID:ENUMDEC EnumDflt:"Illegal Success" */
typedef enum
{
    ATTRIBUTE_NO_CHANGE,            /* EnumTxt:"Attribute Did Not Change"*/
    ATTRIBUTE_CHANGED,              /* EnumTxt:"Attribute Changed"*/
    SERVICE_STARTED,                /* EnumTxt:"Service Started"*/
    NUM_NATIVE_FW_SUCCESSES
} E_NativeFWSuccess;

/* Name: E_NativeFWWarning ClassID:ENUMDEC EnumDflt:"Illegal Warning" */
typedef enum
{
    RESOURCE_IS_CREATED,            /* EnumTxt:"Resource Created"*/
    ATTRIBUTE_UNSUPPORTED,          /* EnumTxt:"Attribute unsupported" */
    FW_WARN_0x03,                   /* <- Insert FW Warnings Here */
    FW_WARN_0x04,
    FW_WARN_0x05,
    FW_WARN_0x06,
    FW_WARN_0x07,

    /* Enum 8 to 16 Reserved for TPU Warnings */
    /* Overwrite TPU_UNDEFINED_WARN entries as Warnings get defined in TPU\Generic\API\TPUuCodeError.h */
    TPU_FW_PWM_PERIOD_LIMIT = 8,       /* EnumTxt:"TPU unable to set desired period (clipped)" */
    TPU_FW_SM_CONST_UNSUPPORTED,       /* EnumTxt:"TPU Self Modifying constant Unsupported" */
    TPU_FW_UNDEFINED_WARN_0x0A,
    TPU_FW_UNDEFINED_WARN_0x0B,
    TPU_FW_UNDEFINED_WARN_0x0C,
    TPU_FW_UNDEFINED_WARN_0x0D,
    TPU_FW_UNDEFINED_WARN_0x0E,
    TPU_FW_UNDEFINED_WARN_0x0F,
    /* End of TPU Warning Reserved Area */

} E_NativeFWWarning;

/* Field1 Definitions to be used in the context of the FRAMEWORK_GLUE_MODULE */
typedef enum
{
    FW_API,

    NUM_FW_SUBSYSTEMS
} E_FWSubSystem;


/* This define can be used to assign a FW error to a E_NativeFWError type variable that then allows the
   debugger (VisionClick) to convert that into the text enum for viewing in a watch window */
#define GetFrameworkErrorCode(in_FWError) (E_NativeFWError)GetErrorCode(in_FWError)

/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: FRAMEWORK_GLUE_MODULE ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: API                   ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: SUCCESS CODES         ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/* Create a FW_API Error Value */
#define CREATE_FW_API_SUCCESS_VALUE(A,B)                 CREATE_SUCCESS_DEFINE_VALUE(FW_API,A,B)

/* Create a FRAMEWORK_GLUE_MODULE Error Value */
#define CREATE_FRAMEWORK_GLUE_SUCCESS_VALUE(A)           CREATE_FW_API_SUCCESS_VALUE(FRAMEWORK_GLUE_MODULE,A)

/* SUCCESS_ATTRIB_NO_CHANGE:                                                                               ::
:: Attempt to modify a framework resource attruibute was made successfully, but the attribute was already  ::
:: at the requested state                                                                                  */
#define SUCCESS_ATTRIB_NO_CHANGE                     CREATE_FRAMEWORK_GLUE_SUCCESS_VALUE(ATTRIBUTE_NO_CHANGE)

/* SUCCESS_ATTRIB_CHANGED:                                                                                 ::
:: Attempt to modify a framework resource attruibute was made successfully, but the attribute was already  ::
:: at the requested state                                                                                  */
#define SUCCESS_ATTRIB_CHANGED                         CREATE_FRAMEWORK_GLUE_SUCCESS_VALUE(ATTRIBUTE_CHANGED)
                                                                                                           
/* SUCCESS_NATIVE_ERROR_SERVICE_STARTED                                                                    ::
:: Attempt to start the Native Error Logging Service was made successfully.                                */
#define SUCCESS_NATIVE_ERROR_SERVICE_STARTED           CREATE_FRAMEWORK_GLUE_SUCCESS_VALUE(SERVICE_STARTED)

/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: FRAMEWORK_GLUE_MODULE ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: API                   ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: WARNING CODES         ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/* Create a FW_API Error Value */
#define CREATE_FW_API_WARNING_VALUE(A,B)                 CREATE_WARNING_DEFINE_VALUE(FW_API,A,B)

/* Create a FRAMEWORK_GLUE_MODULE Error Value */
#define CREATE_FRAMEWORK_GLUE_WARNING_VALUE(A)           CREATE_FW_API_WARNING_VALUE(FRAMEWORK_GLUE_MODULE,A)

/*! \brief Attempt to create a resource that was already created. Call had no detrimental affect upon the system
*/
#define WARNING_RESOURCE_CREATED                     CREATE_FRAMEWORK_GLUE_WARNING_VALUE(RESOURCE_IS_CREATED)

/*! \brief An optional attribute was accessed which is not supported by the particular hardware instance */
#define WARNING_UNSUPPORTED_ATTRIB                 CREATE_FRAMEWORK_GLUE_WARNING_VALUE(ATTRIBUTE_UNSUPPORTED)

/*! An warning that results when an attempt is made to alter an attribute of a resource not created         */
#define WARNING_RESOURCE_NOT_CREATED                CREATE_FRAMEWORK_GLUE_WARNING_VALUE(RESOURCE_NOT_CREATED)

/*! A resource conflict has occured where the behaviour is altered, but still operates */
#define WARNING_RESOURCE_CONFLICT                      CREATE_FRAMEWORK_GLUE_WARNING_VALUE(RESOURCE_CONFLICT)

/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: FRAMEWORK_GLUE_MODULE ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: API                   ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: ERRORS                ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/* Create a FW_API Error Value */
#define CREATE_FW_API_ERROR_VALUE(A,B)                     CREATE_ERROR_DEFINE_VALUE(FW_API,A,B)

/* Create a FRAMEWORK_GLUE_MODULE Error Value */
#define CREATE_FRAMEWORK_GLUE_ERROR_VALUE(A)               CREATE_FW_API_ERROR_VALUE(FRAMEWORK_GLUE_MODULE,A)

/*! ERROR_BAD_RESOURCE_ID
    The ResourceID supplied to perform an operation is illegal                                              */
#define ERROR_BAD_RESOURCE_ID                   CREATE_FRAMEWORK_GLUE_ERROR_VALUE(BAD_RESOURCE_ID)

/* ERROR_MODULE_DOES_NOT_SUPPORT_HARDWARE:                                                                 ::
:: An attempt to access a resource failed because the application is executing on a module that does not   ::
::   have that particular hardware populated                                                               */
#define ERROR_MODULE_DOES_NOT_SUPPORT_HARDWARE  CREATE_FRAMEWORK_GLUE_ERROR_VALUE(HARDWARE_NOT_SUPPORTED)

/* ERROR_BEHAVIOUR_NOT_SUPPORTED:                                                                          ::
:: An attempt was made to use a particular behaviour on a Resource that does not support that behaviour.   ::
::   For example, attempting to set the DutyCycle of ADC input resource                                    */
#define ERROR_BEHAVIOUR_NOT_SUPPORTED           CREATE_FRAMEWORK_GLUE_ERROR_VALUE(BEHAVIOUR_NOT_SUPPORTED)

/* ERROR_BAD_ATTRIBUTE_DATA:                                                                               ::
:: An operation failed because the supplied data was illegal in some way                                   */
#define ERROR_BAD_ATTRIBUTE_DATA                CREATE_FRAMEWORK_GLUE_ERROR_VALUE(BAD_ATTRIBUTE_DATA)

/* ERROR_INTERNAL_OS_FAIL_ON_CREATE:                                                                       ::
:: Describes a generic legacy error that was returned from an internal OS call. Legacy OS implementation   ::
::  uses Boolean error codes that simply indicate success or failure, without any further detail. The most ::
::  likely causes of such an internal failure are: Internal resource conflict because another module       ::
::  resource has already consumed the internal resource, there is insufficient internal resources left to  ::
::  create the resource or the module does not support the resource(ERROR_MODULE_DOES_NOT_SUPPORT_HARDWARE)*/
#define ERROR_INTERNAL_OS_FAIL_ON_CREATE        CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FAIL_ON_CREATE)

/* ERROR_INTERNAL_OS_FAIL_ON_SET_ATTRIBUTE:                                                                ::
:: Describes a generic legacy error that was returned from an internal OS call. Legacy OS implementation   ::
::  uses Boolean error codes that simply indicate success or failure, without any further detail. The most ::
::  likely causes of such an internal failure are: Resource not created or inappropirate data supplied     */
#define ERROR_INTERNAL_OS_FAIL_ON_SET_ATTRIBUTE CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FAIL_ON_SET_ATTRIBUTE)

/* ERROR_RESOURCE_NOT_CREATED:                                                                             ::
:: An error that results when an attempt is made to access a resource that has yet to be created           */
#define ERROR_RESOURCE_NOT_CREATED              CREATE_FRAMEWORK_GLUE_ERROR_VALUE(RESOURCE_NOT_CREATED)

/* ERROR_MISSING_ATTRIBUTE:                                                                                ::
:: An attribute set or Resource creation was attempted without assigning a value to a required attribute   ::
:: Supply the attribute and set that attribute as being valid in the attribute mask                        */
#define ERROR_MISSING_ATTRIBUTE                 CREATE_FRAMEWORK_GLUE_ERROR_VALUE(MISSING_ATTRIBUTE)

/* ERROR_BEHAVIOUR_METHOD_UNSUPPORTED:                                                                     ::
:: A method call was made on a resource that does not support that method. For example, some resources     ::
::   can't be destroyed once created.                                                                      */
#define ERROR_BEHAVIOUR_METHOD_UNSUPPORTED      CREATE_FRAMEWORK_GLUE_ERROR_VALUE(BEHAVIOUR_METHOD_UNSUPPORTED)

/* ERROR_INTERNAL_OS_FAIL_ON_OPERATION:                                                                    ::
:: Describes a generic legacy error that was returned from an internal OS call. Legacy OS implementation   ::
::  uses Boolean error codes that simply indicate success or failure, without any further detail. The most ::
::  likely causes of such an internal failure are: Resource not created or inappropirate data supplied     ::
:: Different to an ERROR_INTERNAL_OS_FAIL_ON_SET_ATTRIBUTE in that an expected operation was to result     ::
::  (like a notfication callback) but will not because of the failure                                      */
#define ERROR_INTERNAL_OS_FAIL_ON_OPERATION     CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FAIL_ON_OPERATION)

/* ERROR_INTERNAL_OS_FAIL_ON_DESTROY:                                                                      ::
:: Describes a generic legacy error that was returned from an internal OS call. Legacy OS implementation   ::
::  uses Boolean error codes that simply indicate success or failure, without any further detail. The most ::
::  likely causes of such an internal failure like this is the destroy method not being supported by the   ::
::  underlyng OS. Some resources never had their Destroy methods implemented.                              */
#define ERROR_INTERNAL_OS_FAIL_ON_DESTROY       CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FAIL_ON_DESTROY)

/* ERROR_RESOURCE_CREATED:                                                                                 ::
:: An error that results when an attempt to create a resource fails because that resource already exists   */
#define ERROR_RESOURCE_CREATED                  CREATE_FRAMEWORK_GLUE_ERROR_VALUE(CREATED_RESOURCE)

/* ERROR_RESOURCE_CONFLICT:                                                                                ::
:: An error that results when an attempt to create a resource fails because another mutually exclusive     ::
::   behaviour has already been mapped to that resource                                                    */
#define ERROR_RESOURCE_CONFLICT                 CREATE_FRAMEWORK_GLUE_ERROR_VALUE(RESOURCE_CONFLICT)

/* ERROR_UNKNOWN_CRANK_ENCODER:                                                                            ::
:: Attempt to create an encoder failed because the supplied crank encoder attribute was unknown and likely ::
::   an illegal or out of range value                                                                      */
#define ERROR_UNKNOWN_CRANK_ENCODER             CREATE_FRAMEWORK_GLUE_ERROR_VALUE(UNKNOWN_CRANK_ENCODER)

/* ERROR_2STROKE_UNSUPPORTED:                                                                              ::
:: The attempt to create an encoder failed because the framework does not support 2-stroke operation for   ::
::   the selected encoder. A framework upgrade is required before this behaviour can be supported          */
#define ERROR_2STROKE_UNSUPPORTED               CREATE_FRAMEWORK_GLUE_ERROR_VALUE(ENCODER_2STROKE_UNSUPPORTED)

/* ERROR_720_UNSUPPORTED:                                                                                  ::
:: The attempt to create the encoder failed because the selected crank encoder does not support a 720degCA ::
::   version of the pattern in this framework implementation. Either this attribute has been set in error  ::
::   or a framework upgrade is required.                                                                   */
#define ERROR_720_UNSUPPORTED                   CREATE_FRAMEWORK_GLUE_ERROR_VALUE(ENCODER_720_UNSUPPORTED)

/* ERROR_UNSUPPORTED_CAM_ENCODER:                                                                          ::
:: The framework does not support the requested Crank/Cam Encoder pair.                                    */
#define ERROR_UNSUPPORTED_CAM_ENCODER           CREATE_FRAMEWORK_GLUE_ERROR_VALUE(UNSUPPORTED_CAM_ENCODER)

/* ERROR_UNKNOWN_CAM_ENCODER:                                                                              ::
:: Attempt to create an encoder failed because the supplied cam encoder attribute was unknown and likely   ::
::   an illegal or out of range value                                                                      */
#define ERROR_UNKNOWN_CAM_ENCODER               CREATE_FRAMEWORK_GLUE_ERROR_VALUE(UNKNOWN_CAM_ENCODER)

/* ERROR_NOT_ZERO_SPEED:                                                                                   ::
:: Attempt to perform an operation that is not allowed while there is engine speed. Failure resulted.      */
#define ERROR_NOT_ZERO_SPEED                    CREATE_FRAMEWORK_GLUE_ERROR_VALUE(NOT_ZERO_SPEED)

/* ERROR_MODULE_NOT_SUPPORT_HW_INTERFACE                                                                   ::
:: Error resulted because a signal interface for particular module hardware was not supported. For example ::
::   requesting a VR signal interface when the hardware only support a digital interface                   */
#define ERROR_MODULE_NOT_SUPPORT_HW_INTERFACE   CREATE_FRAMEWORK_GLUE_ERROR_VALUE(HW_INTERFACE_NOT_SUPPORT)

/* ERROR_NATIVE_ERROR_SERVICE_ALREADY_STARTED                                                              ::
:: Error result because attempting to start a service that was previously started.                         */
#define ERROR_NATIVE_ERROR_SERVICE_ALREADY_STARTED CREATE_FRAMEWORK_GLUE_ERROR_VALUE(SERVICE_ALREADY_STARTED)

/* ERROR_REGISTER_WRITE_VERIFY_FAILED
:: An attempt to access a framework register failed to verify.                                             */
#define ERROR_REGISTER_WRITE_VERIFY_FAILED    CREATE_FRAMEWORK_GLUE_ERROR_VALUE(REGISTER_WRITE_VERIFY_FAILED)

/* ERROR_REGISTER_ACCESS_DENIED
:: As attempt to access a framework register failed because of insufficient privledges (read-only) or the
::   register requested was a bad register                                                                 */
#define ERROR_REGISTER_ACCESS_DENIED               CREATE_FRAMEWORK_GLUE_ERROR_VALUE(REGISTER_ACCESS_DENIED)

/* ERROR_OUT_OF_MEMORY
An attempt to create or employ a resource that uses the free store memory (heap) failed because there was 
  insufficient memory */
#define ERROR_OUT_OF_MEMORY                        CREATE_FRAMEWORK_GLUE_ERROR_VALUE(OUT_OF_MEMORY)

/* ERROR_SW_Q_OVERRUN
   A software Q has suffered from an overrun condition */
#define ERROR_SW_Q_OVERRUN                              CREATE_FRAMEWORK_GLUE_ERROR_VALUE(SW_Q_OVERRUN)

/* ERROR_HW_Q_OVERRUN
   A hardware Q has suffered from an overrun condition */
#define ERROR_HW_Q_OVERRUN                              CREATE_FRAMEWORK_GLUE_ERROR_VALUE(HW_Q_OVERRUN)

/* ERROR_UNSUPPORTED_IN_ARCHIVE
   A request to use a resource or service failed because this particular build of the archive does not
   support it. This error is returned for systems that are likely supported by a different archive configuration */
#define ERROR_UNSUPPORTED_IN_ARCHIVE          CREATE_FRAMEWORK_GLUE_ERROR_VALUE(UNSUPPORTED_IN_ARCHIVE)

/* ERROR_OUT_OF_RESOURCE
   An operation was unable to acquire sufficient internal framework resources to succeed. Some implementations
   of a framework may places limits on the number of a type of resource that can be declared, or may require
   internal hardware facilities to be free before the operation could work. */
#define ERROR_OUT_OF_RESOURCE                          CREATE_FRAMEWORK_GLUE_ERROR_VALUE(OUT_OF_RESOURCE)

/* ERROR_OS_RESOURCE_CONFLICT
   The framework has determined that the resource was free for use, but resources required by the OS to
   implement that resource were not available. Error probably due to an internal OS resource mapping problem */
#define ERROR_OS_RESOURCE_CONFLICT                     CREATE_FRAMEWORK_GLUE_ERROR_VALUE(OS_RESOURCE_CONFLICT)

/*! \brief The driver mode selected for a mixed driver resource conflicts with a previous driver mode assignment

    Some resources have mixed driver capabilities. For example RES_EST13 through RES_EST16 can be configured
    through software at runtime to operate in a low side driver mode or a push pull mode, but all the drivers
    can only be configured in one mode of the other. This error results if the mode is mixed */
#define ERROR_DRIVER_CONFLICT                          CREATE_FRAMEWORK_GLUE_ERROR_VALUE(DRIVER_CONFLICT)

/*! \brief The realtime clock hardware has been without power and the time is unknown. */
#define ERROR_RTC_TIME_ILLEGAL                         CREATE_FRAMEWORK_GLUE_ERROR_VALUE(RTC_TIME_ILLEGAL)

/*! \brief The operation failed because it was made from an ISR context, which does not allow the
    requested operation. Typically this is because the operation may have to wait for the resource
    to become available before it can continue, which is not allowed in an interrupt */
#define ERROR_ACCESS_FROM_ISR                          CREATE_FRAMEWORK_GLUE_ERROR_VALUE(ACCESS_FROM_ISR)

/*! \brief The Flash File System (FFS) had an internal error */
#define ERROR_FFS_INTERNAL                             CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FFS_INTERNAL)

/*! \brief The file system media for the target file operation is not mounted or failed to mount.

           If the media is internal to the module then this error describes the situation where some form or failure,
           be it hardware or software has prevented the media from mounting. For external media, this error describes
           the situation where the media associated with the file operation is not mounted. Attempting to access a
           USB mass storage device that was not connected would give this error. */
#define ERROR_FFS_UNMOUNTED                            CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FFS_UNMOUNTED)

/*! \brief The requested operation timed-out before completing */
#define ERROR_REQUEST_TIMEOUT                          CREATE_FRAMEWORK_GLUE_ERROR_VALUE(REQUEST_TIMEOUT)

/*! \brief A task kernel API has been used in an inappropriate way */
#define ERROR_TASK_KERNEL_API                          CREATE_FRAMEWORK_GLUE_ERROR_VALUE(TASK_KERNEL_API)

/*! \brief The operation was aborted. For example, waiting upon a semaphore would return this error if
           the semaphore being waited upon was destroyed */
#define ERROR_REQUEST_ABORTED                          CREATE_FRAMEWORK_GLUE_ERROR_VALUE(REQUEST_ABORTED)

/*! \brief The operation failed because the specified file could not be found */
#define ERROR_FILE_NOT_FOUND                           CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FILE_NOT_FOUND)

/*! \brief The specified file name is illegal because it contains illegal characters */
#define ERROR_FILENAME_INVALID                         CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FILENAME_INVALID)

/*! \brief The specified file name is not unique */
#define ERROR_FILENAME_NOT_UNIQUE                      CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FILENAME_NOT_UNIQUE)

/*! \brief The filename (or its fully qualified path equivalent) is too long for the file system to handle */
#define ERROR_FILENAME_TOO_LONG                        CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FILENAME_TOO_LONG)

/*! \brief An attempt to remove a directory failed because that directory was not empty */
#define ERROR_DIRECTORY_NOT_EMPTY                      CREATE_FRAMEWORK_GLUE_ERROR_VALUE(DIRECTORY_NOT_EMPTY)

/*! \brief An attempt to remove a directory failed because that directory was not empty */
#define ERROR_DIRECTORY_NAME_NOT_FOUND                 CREATE_FRAMEWORK_GLUE_ERROR_VALUE(DIRECTORY_NAME_NOT_FOUND)

/*! \brief The directory name contains invalid characters */
#define ERROR_DIRECTORY_INVALID                        CREATE_FRAMEWORK_GLUE_ERROR_VALUE(DIRECTORY_INVALID)

/*! \brief The directory already exists */
#define ERROR_DIRECTORY_EXIST                          CREATE_FRAMEWORK_GLUE_ERROR_VALUE(DIRECTORY_EXIST)

/*! \brief The directory name when extended to a fully qualified path is too long for the file system to handle */
#define ERROR_DIRECTORY_TOO_DEEP                       CREATE_FRAMEWORK_GLUE_ERROR_VALUE(DIRECTORY_TOO_DEEP)

/*! \brief A supplied handle was not legal */
#define ERROR_BAD_FW_HANDLE                            CREATE_FRAMEWORK_GLUE_ERROR_VALUE(BAD_FW_HANDLE)

/*! \brief A file operation failed or did not completely succeed (reading fewer bytes than requested) because
           the end of the file was reached */
#define ERROR_FS_EOF                                   CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FS_EOF)

/*! \brief The file system media that a file operation was conducted upon failed because the media was full.
           What media is dependant upon the targeted volume */
#define ERROR_FS_MEDIA_FULL                            CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FS_MEDIA_FULL)

/*! \brief The file system media that a file operation was conducted upon failed because the media would not write. */
#define ERROR_FS_MEDIA_WRITE                           CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FS_MEDIA_WRITE)

/*! \brief The file system media that a file operation was conducted upon failed because the media would not read. */
#define ERROR_FS_MEDIA_READ                            CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FS_MEDIA_READ)

/*! \brief The file system media that a file operation was conducted upon failed because the media was write protected. */
#define ERROR_FS_MEDIA_WRITE_PROTECT                   CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FS_MEDIA_WRITE_PROTECT)

/*! \brief The file operation failed because the file was opened for read only or has its read only attribute set */
#define ERROR_FS_FILE_READ_ONLY                        CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FS_FILE_READ_ONLY)

/*! \brief The file operation failed because the target media has remounted. The media may be the same or a different
           volume */
#define ERROR_FS_MEDIA_CHANGED                         CREATE_FRAMEWORK_GLUE_ERROR_VALUE(FS_MEDIA_CHANGED)

/*! \brief There was a catistrophic failure within the USB stack */
#define ERROR_USB_INTERNAL                             CREATE_FRAMEWORK_GLUE_ERROR_VALUE(USB_INTERNAL)

/*! \brief The system voltage is too high for the USB IC to allow operation
*/
#define ERROR_USB_HIGH_VOLTAGE                         CREATE_FRAMEWORK_GLUE_ERROR_VALUE(USB_HIGH_VOLTAGE)

/*! \brief The encoder has not been created but the requested operation requires the encoder to exist
*/
#define ERROR_ENCODER_NOT_CREATED                      CREATE_FRAMEWORK_GLUE_ERROR_VALUE(ENCODER_NOT_CREATED)

#pragma pack()
/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/

/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/

NativeError_S StartNativeErrorLoggingService(E_ErrorModule in_eModule, S_SetDisplayInfo*);
NativeBool IsNativeErrorLogged(NativeError_S in_NativeError);

/* Define the logging of an incorrect FW (Framework) Usage Error as nothing. Can later provide an implementation */
#define LogOSUsageError(a,b,c)  

#endif /* __FRAMEWORKERROR_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
